import { execSync } from 'node:child_process';
import { d as defineCommand, c as consola, b as colors } from '../shared/nuxi.4fde776c.mjs';
import 'node:util';
import 'node:path';
import 'node:process';
import 'node:tty';
import { l as loadKit, r as readPackageJSON } from '../shared/nuxi.0c472721.mjs';
import { g as getPackageManager, p as packageManagerLocks } from '../shared/nuxi.18a5211a.mjs';
import { r as rmRecursive, t as touchFile } from '../shared/nuxi.976cd438.mjs';
import { c as cleanupNuxtDirs, n as nuxtVersionToGitIdentifier } from '../shared/nuxi.6344d342.mjs';
import { s as sharedArgs, l as legacyRootDirArgs, r as resolve, a as relative } from '../shared/nuxi.5c071e73.mjs';
import 'node:url';
import '../shared/nuxi.70ab727b.mjs';
import 'node:module';
import 'node:fs';
import '../shared/nuxi.cda6864c.mjs';
import 'node:assert';
import 'node:v8';
import 'node:perf_hooks';
import '../shared/nuxi.0508e766.mjs';
import './satisfies.mjs';
import '../shared/nuxi.2155838d.mjs';
import '../shared/nuxi.cc8dd4a9.mjs';
import '../shared/nuxi.b3d5700f.mjs';
import 'crypto';
import 'fs';
import 'module';
import 'path';
import 'perf_hooks';
import 'os';
import 'vm';
import 'url';
import 'assert';
import 'process';
import 'v8';
import 'util';
import 'tty';
import 'stream';
import 'events';
import 'node:stream';

async function getNuxtVersion(path) {
  try {
    const pkg = await readPackageJSON("nuxt", { url: path });
    if (!pkg.version) {
      consola.warn("Cannot find any installed nuxt versions in ", path);
    }
    return pkg.version || null;
  } catch {
    return null;
  }
}
const upgrade = defineCommand({
  meta: {
    name: "upgrade",
    description: "Upgrade nuxt"
  },
  args: {
    ...sharedArgs,
    ...legacyRootDirArgs,
    force: {
      type: "boolean",
      alias: "f",
      description: "Force upgrade to recreate lockfile and node_modules"
    }
  },
  async run(ctx) {
    const cwd = resolve(ctx.args.cwd || ctx.args.rootDir || ".");
    const packageManager = getPackageManager(cwd);
    if (!packageManager) {
      consola.error(
        `Unable to determine the package manager used by this project.

No lock files found in \`${cwd}\`, and no \`packageManager\` field specified in \`package.json\`.

Please either add the \`packageManager\` field to \`package.json\` or execute the installation command for your package manager. For example, you can use \`pnpm i\`, \`npm i\`, \`bun i\`, or \`yarn i\`, and then try again.`
      );
      process.exit(1);
    }
    const packageManagerVersion = execSync(`${packageManager} --version`).toString("utf8").trim();
    consola.info("Package Manager:", packageManager, packageManagerVersion);
    const currentVersion = await getNuxtVersion(cwd) || "[unknown]";
    consola.info("Current nuxt version:", currentVersion);
    const pmLockFile = resolve(cwd, packageManagerLocks[packageManager]);
    const forceRemovals = ["node_modules", relative(process.cwd(), pmLockFile)].map((p) => colors.cyan(p)).join(" and ");
    if (ctx.args.force === void 0) {
      ctx.args.force = await consola.prompt(
        `Would you like to recreate ${forceRemovals} to fix problems with hoisted dependency versions and ensure you have the most up-to-date dependencies?`,
        {
          type: "confirm",
          default: true
        }
      );
    }
    if (ctx.args.force) {
      consola.info(
        `Recreating ${forceRemovals}. If you encounter any issues, revert the changes and try with \`--no-force\``
      );
      await rmRecursive([pmLockFile, resolve(cwd, "node_modules")]);
      await touchFile(pmLockFile);
    }
    consola.info("Installing latest Nuxt 3 release...");
    execSync(
      `${packageManager} ${packageManager === "yarn" ? "add" : "install"} -D nuxt`,
      { stdio: "inherit", cwd }
    );
    let buildDir = ".nuxt";
    try {
      const { loadNuxtConfig } = await loadKit(cwd);
      const nuxtOptions = await loadNuxtConfig({ cwd });
      buildDir = nuxtOptions.buildDir;
    } catch {
    }
    await cleanupNuxtDirs(cwd, buildDir);
    const upgradedVersion = await getNuxtVersion(cwd) || "[unknown]";
    consola.info("Upgraded nuxt version:", upgradedVersion);
    if (upgradedVersion === currentVersion) {
      consola.success("You're already using the latest version of nuxt.");
    } else {
      consola.success(
        "Successfully upgraded nuxt from",
        currentVersion,
        "to",
        upgradedVersion
      );
      const commitA = nuxtVersionToGitIdentifier(currentVersion);
      const commitB = nuxtVersionToGitIdentifier(upgradedVersion);
      if (commitA && commitB) {
        consola.info(
          "Changelog:",
          `https://github.com/nuxt/nuxt/compare/${commitA}...${commitB}`
        );
      }
    }
  }
});

export { upgrade as default };
