import { customRef, getCurrentScope, nextTick, onScopeDispose, ref, watch } from "vue";
import { parse, serialize } from "cookie-es";
import { deleteCookie, getCookie, getRequestHeader, setCookie } from "h3";
import destr from "destr";
import { isEqual } from "ohash";
import { useNuxtApp } from "../nuxt.js";
import { useRequestEvent } from "./ssr.js";
const CookieDefaults = {
  path: "/",
  watch: true,
  decode: (val) => destr(decodeURIComponent(val)),
  encode: (val) => encodeURIComponent(typeof val === "string" ? val : JSON.stringify(val))
};
export function useCookie(name, _opts) {
  const opts = { ...CookieDefaults, ..._opts };
  const cookies = readRawCookies(opts) || {};
  let delay;
  if (opts.maxAge !== void 0) {
    delay = opts.maxAge * 1e3;
  } else if (opts.expires) {
    delay = opts.expires.getTime() - Date.now();
  }
  const hasExpired = delay !== void 0 && delay <= 0;
  const cookieValue = hasExpired ? void 0 : cookies[name] ?? opts.default?.();
  const cookie = import.meta.client && delay && !hasExpired ? cookieRef(cookieValue, delay) : ref(cookieValue);
  if (import.meta.dev && hasExpired) {
    console.warn(`[nuxt] not setting cookie \`${name}\` as it has already expired.`);
  }
  if (import.meta.client) {
    const channel = typeof BroadcastChannel === "undefined" ? null : new BroadcastChannel(`nuxt:cookies:${name}`);
    const callback = () => {
      writeClientCookie(name, cookie.value, opts);
      channel?.postMessage(opts.encode(cookie.value));
    };
    let watchPaused = false;
    if (getCurrentScope()) {
      onScopeDispose(() => {
        watchPaused = true;
        callback();
        channel?.close();
      });
    }
    if (channel) {
      channel.onmessage = (event) => {
        watchPaused = true;
        cookie.value = opts.decode(event.data);
        nextTick(() => {
          watchPaused = false;
        });
      };
    }
    if (opts.watch) {
      watch(
        cookie,
        () => {
          if (watchPaused) {
            return;
          }
          callback();
        },
        { deep: opts.watch !== "shallow" }
      );
    } else {
      callback();
    }
  } else if (import.meta.server) {
    const nuxtApp = useNuxtApp();
    const writeFinalCookieValue = () => {
      if (!isEqual(cookie.value, cookies[name])) {
        writeServerCookie(useRequestEvent(nuxtApp), name, cookie.value, opts);
      }
    };
    const unhook = nuxtApp.hooks.hookOnce("app:rendered", writeFinalCookieValue);
    nuxtApp.hooks.hookOnce("app:error", () => {
      unhook();
      return writeFinalCookieValue();
    });
  }
  return cookie;
}
function readRawCookies(opts = {}) {
  if (import.meta.server) {
    return parse(getRequestHeader(useRequestEvent(), "cookie") || "", opts);
  } else if (import.meta.client) {
    return parse(document.cookie, opts);
  }
}
function serializeCookie(name, value, opts = {}) {
  if (value === null || value === void 0) {
    return serialize(name, value, { ...opts, maxAge: -1 });
  }
  return serialize(name, value, opts);
}
function writeClientCookie(name, value, opts = {}) {
  if (import.meta.client) {
    document.cookie = serializeCookie(name, value, opts);
  }
}
function writeServerCookie(event, name, value, opts = {}) {
  if (event) {
    if (value !== null && value !== void 0) {
      return setCookie(event, name, value, opts);
    }
    if (getCookie(event, name) !== void 0) {
      return deleteCookie(event, name, opts);
    }
  }
}
const MAX_TIMEOUT_DELAY = 2147483647;
function cookieRef(value, delay) {
  let timeout;
  let elapsed = 0;
  if (getCurrentScope()) {
    onScopeDispose(() => {
      clearTimeout(timeout);
    });
  }
  return customRef((track, trigger) => {
    function createExpirationTimeout() {
      clearTimeout(timeout);
      const timeRemaining = delay - elapsed;
      const timeoutLength = timeRemaining < MAX_TIMEOUT_DELAY ? timeRemaining : MAX_TIMEOUT_DELAY;
      timeout = setTimeout(() => {
        elapsed += timeoutLength;
        if (elapsed < delay) {
          return createExpirationTimeout();
        }
        value = void 0;
        trigger();
      }, timeoutLength);
    }
    return {
      get() {
        track();
        return value;
      },
      set(newValue) {
        createExpirationTimeout();
        value = newValue;
        trigger();
      }
    };
  });
}
